from __future__ import print_function, division, absolute_import

import argparse

import numpy as np

from libmmc import (sample_correlation_matrix, get_clustering,
        get_corr_fig2, get_corr_fig2_desired_clustering,
        get_clustering_accuracy)


def main(args):
    n = args.observations
    nsamples = args.samples

    # Consider all values of tuning parameter sigma in this array.
    sigmas, step = np.linspace(args.sigma_low, args.sigma_high,
            num=args.sigma_num, retstep=True)

    R = get_corr_fig2(
            expansion_factor=args.expansion_factor)
    desired_v = get_corr_fig2_desired_clustering(
            expansion_factor=args.expansion_factor)
    totals = np.zeros(3, dtype=float)
    total_modularity = 0
    best_sigmas = []
    if n < 0:
        nsamples = 1
        observed_v, sigma, m = get_clustering(
                R, sigmas, verbose=args.verbose)
        total_modularity += m
        best_sigmas.append(sigma)
        accuracies = get_clustering_accuracy(observed_v, desired_v)
        totals += np.array(accuracies)
    else:
        for i in range(nsamples):
            C = sample_correlation_matrix(R, n)
            observed_v, sigma, m = get_clustering(
                    C, sigmas, verbose=args.verbose)
            total_modularity += m
            best_sigmas.append(sigma)
            accuracies = get_clustering_accuracy(observed_v, desired_v)
            totals += np.array(accuracies)
    print(totals / nsamples)
    print(total_modularity / nsamples)


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument('--observations', type=int, default=9)
    parser.add_argument('--samples', type=int, default=10000)
    parser.add_argument('--expansion-factor', type=int, default=1,
            help=(
                'Expand the number of variables in the analysis '
                'by this factor.'))
    parser.add_argument('--sigma-low', type=float, default=0.05,
            help='Low value of sigma (Default: 0.05).')
    parser.add_argument('--sigma-high', type=float, default=0.35,
            help='High value of sigma (Default: 0.35).')
    parser.add_argument('--sigma-num', type=float, default=7,
            help='Number of values of sigma to search (Default: 7).')
    parser.add_argument('--verbose', '-v', action='store_true',
            help='Show more information on the screen.')
    args = parser.parse_args()
    main(args)
